<?php

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;
use App\Http\Controllers\API\Auth\AuthController;
use App\Http\Controllers\API\Tours\TourController as ApiTourController;
use App\Http\Controllers\Admin\TourController;
use App\Http\Controllers\API\Reservations\ReservationController;
use App\Http\Controllers\API\Payments\PaymentController;
use App\Http\Controllers\API\Payments\PaymentLinkController;
use App\Http\Controllers\API\Customers\CustomerPanelController;
use App\Http\Controllers\API\Customers\DocumentController;
use App\Http\Controllers\API\PageController;
use App\Http\Controllers\Admin\ReservationApprovalController;
use App\Http\Controllers\Admin\ReservationManagementController;
use App\Http\Controllers\Admin\DashboardController;
use App\Http\Controllers\Admin\ImageController;
use App\Http\Controllers\Admin\StoryController;
use App\Http\Controllers\Admin\TestimonialController;
use App\Http\Controllers\Admin\BrandController;
use App\Http\Controllers\Admin\SettingsController;
use App\Http\Controllers\Webhooks\StripeWebhookController;
use App\Http\Controllers\Api\Errors\ErrorLogController;
use App\Http\Controllers\Admin\ExtraController;
use App\Http\Controllers\API\GYG\GYGController;
use App\Http\Controllers\Admin\CashDeskController;
use App\Http\Controllers\Admin\SliderController;
use App\Http\Controllers\Api\Villa\VillaController;
use App\Http\Controllers\Api\Hotel\HotelController;
use App\Http\Controllers\API\Invoices\InvoiceController;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
|
| Here is where you can register API routes for your application. These
| routes are loaded by the RouteServiceProvider and all of them will
| be assigned to the "api" middleware group. Make something great!
|
*/

// GetYourGuide Integration Routes
Route::prefix('gyg')->group(function () {
    Route::get('/availability', [GYGController::class, 'availability']);
    Route::post('/bookings', [GYGController::class, 'booking']);
});

// Health Check
Route::get('/health', function () {
    return response()->json([
        'status' => 'ok',
        'timestamp' => now()->toIso8601String(),
        'version' => '1.0.0'
    ]);
});

// Public Routes
Route::prefix('auth')->group(function () {
    Route::post('/register', [AuthController::class, 'register']);
    Route::post('/login', [AuthController::class, 'login']);
    Route::post('/forgot-password', [AuthController::class, 'forgotPassword']);
    Route::post('/reset-password', [AuthController::class, 'resetPassword']);
});

// Public Payment Routes (for Stripe public key + checkout from cart)
Route::get('/payments/stripe-key', [PaymentController::class, 'getPublicKey']);
Route::post('/checkout/pre-reservation', [PaymentController::class, 'preReservation']);
Route::post('/checkout/from-cart', [PaymentController::class, 'fromCart']);
Route::post('/checkout/confirm-from-cart', [PaymentController::class, 'confirmFromCart']);

// Public Tour Routes (liste ve detay görüntüleme)
Route::get('/tours', [ApiTourController::class, 'index']);
Route::get('/tours/{id}', [ApiTourController::class, 'show']);
Route::get('/tours/{id}/dates', [ApiTourController::class, 'dates']);
Route::get('/occupied-seats', [ApiTourController::class, 'occupiedSeats']);

// Public Activity Routes
Route::prefix('activities')->group(function () {
    Route::get('/', [\App\Http\Controllers\Api\Activities\ActivityController::class, 'index']);
    Route::get('/{id}', [\App\Http\Controllers\Api\Activities\ActivityController::class, 'show']);
    Route::get('/{id}/dates', [\App\Http\Controllers\Api\Activities\ActivityController::class, 'dates']);
    Route::get('/{id}/pricing', [\App\Http\Controllers\Api\Activities\ActivityController::class, 'pricing']);
});

// Public Visa Routes
Route::prefix('visas')->group(function () {
    Route::get('/', [\App\Http\Controllers\Api\Visas\VisaController::class, 'index']);
    Route::get('/{slug}', [\App\Http\Controllers\Api\Visas\VisaController::class, 'show']);
    Route::post('/apply', [\App\Http\Controllers\Api\Visas\VisaController::class, 'storeApplication']);
});

// Public Inquiry Route
Route::post('/inquiries', [\App\Http\Controllers\API\Inquiries\InquiryController::class, 'store']);

// Admin Tour Routes - Geçici olarak public (test için)
Route::get('/admin/tours', [TourController::class, 'index']);
Route::get('/admin/tours/{id}', [TourController::class, 'show']);
Route::post('/admin/tours', [TourController::class, 'store']);
Route::put('/admin/tours/{id}', [TourController::class, 'update']);
Route::delete('/admin/tours/{id}', [TourController::class, 'destroy']);

// Admin Activity Routes
Route::get('/admin/activities', [\App\Http\Controllers\Admin\ActivityController::class, 'index']);
Route::get('/admin/activities/{id}', [\App\Http\Controllers\Admin\ActivityController::class, 'show']);
Route::post('/admin/activities', [\App\Http\Controllers\Admin\ActivityController::class, 'store']);
Route::put('/admin/activities/{id}', [\App\Http\Controllers\Admin\ActivityController::class, 'update']);
Route::delete('/admin/activities/{id}', [\App\Http\Controllers\Admin\ActivityController::class, 'destroy']);
Route::post('/admin/activities/{id}/pricing-tiers', [\App\Http\Controllers\Admin\ActivityController::class, 'storePricingTier']);
Route::delete('/admin/activities/{id}/pricing-tiers/{tierId}', [\App\Http\Controllers\Admin\ActivityController::class, 'destroyPricingTier']);

// Admin Activity Date Routes
Route::post('/admin/activities/{id}/dates', [\App\Http\Controllers\Admin\ActivityDateController::class, 'store']);
Route::put('/admin/activities/{activityId}/dates/{id}', [\App\Http\Controllers\Admin\ActivityDateController::class, 'update']);
Route::delete('/admin/activities/{activityId}/dates/{id}', [\App\Http\Controllers\Admin\ActivityDateController::class, 'destroy']);

// Public Reservation Inquiry
Route::get('/reservations/easy-code/{easyCode}', [ReservationController::class, 'getByEasyCode']);

// Public Reservation Create (guest checkout – optional auth)
Route::post('/reservations', [ReservationController::class, 'store']);

// Public Page Routes (Dynamic Pages)
Route::get('/pages', [PageController::class, 'index']);
Route::get('/pages/{slug}', [PageController::class, 'show']);

// Error Logging (Public - for frontend error reporting)
Route::post('/errors/log', [ErrorLogController::class, 'log']);

// Protected Routes (Authentication Required)
Route::middleware('auth:sanctum')->group(function () {
    
    // Authentication
    Route::prefix('auth')->group(function () {
        Route::get('/user', [AuthController::class, 'user']);
        Route::post('/logout', [AuthController::class, 'logout']);
        Route::post('/refresh', [AuthController::class, 'refresh']);
    });

    // Tours (Authenticated)
    Route::prefix('tours')->group(function () {
        Route::get('/{id}/availability', [TourController::class, 'availability']);
        Route::get('/{id}/pricing', [TourController::class, 'pricing']);
    });

    // Reservations (create is public above; auth only for list/show/cancel/status)
    Route::prefix('reservations')->group(function () {
        Route::get('/', [ReservationController::class, 'index']);
        Route::get('/{id}', [ReservationController::class, 'show']);
        Route::patch('/{id}/cancel', [ReservationController::class, 'cancel']);
        Route::get('/{id}/status', [ReservationController::class, 'status']);
    });

    // Payments
    Route::prefix('payments')->group(function () {
        Route::post('/create-link', [PaymentLinkController::class, 'create']);
        Route::post('/checkout', [PaymentController::class, 'checkout']);
        Route::post('/create-payment-intent', [PaymentController::class, 'createPaymentIntent']);
        Route::post('/confirm', [PaymentController::class, 'confirmPayment']);
        Route::get('/{id}/status', [PaymentController::class, 'status']);
        Route::post('/{id}/retry', [PaymentController::class, 'retry']);
    });

    // Customer Panel
    Route::prefix('customer')->group(function () {
        Route::get('/dashboard', [CustomerPanelController::class, 'dashboard']);
        Route::get('/reservations', [CustomerPanelController::class, 'reservations']);
        Route::get('/reservations/{id}', [CustomerPanelController::class, 'showReservation']);
        Route::patch('/reservations/{id}/guests/{guestId}/meal-selections', [CustomerPanelController::class, 'updateGuestMealSelections']);
        Route::get('/payments', [CustomerPanelController::class, 'payments']);
        Route::get('/documents/{reservationId}', [DocumentController::class, 'index']);
        Route::get('/documents/{reservationId}/download/{type}', [DocumentController::class, 'download']);
        Route::get('/messages', [CustomerPanelController::class, 'messages']);
        Route::post('/messages', [CustomerPanelController::class, 'sendMessage']);
    });
});

// Admin Routes - Authentication ve Admin yetkisi gerekli
Route::middleware(['auth:sanctum', 'admin'])->prefix('admin')->group(function () {
    
    // Dashboard
    Route::get('/dashboard', [DashboardController::class, 'index']);
    
    // Reservation Management
    Route::prefix('reservations')->group(function () {
        Route::get('/pending', [ReservationManagementController::class, 'pending']);
        Route::get('/all', [ReservationManagementController::class, 'index']);
        Route::get('/for-tour-date', [ReservationManagementController::class, 'forTourDate']);
        Route::post('/bulk-approve', [ReservationApprovalController::class, 'bulkApprove']);
        Route::post('/bulk-reject', [ReservationApprovalController::class, 'bulkReject']);
        Route::get('/{id}', [ReservationManagementController::class, 'show']);
        Route::patch('/{id}', [ReservationManagementController::class, 'update']);
        Route::post('/{id}/approve', [ReservationApprovalController::class, 'approve']);
        Route::post('/{id}/reject', [ReservationApprovalController::class, 'reject']);
        Route::post('/{id}/payment', [ReservationManagementController::class, 'addPayment']);
        Route::get('/{id}/email-preview', [ReservationManagementController::class, 'emailPreview']);
        Route::post('/{id}/send-email', [ReservationManagementController::class, 'sendEmail']);
        Route::post('/{id}/documents/regenerate', [ReservationManagementController::class, 'regenerateDocuments']);
        Route::post('/{id}/proforma', [ReservationManagementController::class, 'createProforma']);
        Route::post('/{id}/guests', [ReservationManagementController::class, 'storeGuest']);
        Route::patch('/{id}/guests/{guestId}', [ReservationManagementController::class, 'updateGuest']);
        Route::delete('/{id}/guests/{guestId}', [ReservationManagementController::class, 'destroyGuest']);
    });

    // Tour Management
    Route::get('/tours', [TourController::class, 'index']);
    Route::get('/tours/{id}', [TourController::class, 'show']);
    Route::post('/tours', [TourController::class, 'store']);
    Route::put('/tours/{id}', [TourController::class, 'update']);
    Route::delete('/tours/{id}', [TourController::class, 'destroy']);
    
    // Image Upload
    Route::post('/images/upload', [ImageController::class, 'upload']);

    // Extras & Merchandise (global list – tur dışında yönetim)
    Route::get('/extras', [ExtraController::class, 'index']);
    Route::get('/extras/{id}', [ExtraController::class, 'show']);
    Route::post('/extras', [ExtraController::class, 'store']);
    Route::put('/extras/{id}', [ExtraController::class, 'update']);
    Route::delete('/extras/{id}', [ExtraController::class, 'destroy']);

    // Brands Management
    Route::get('/brands', [BrandController::class, 'index']);
    Route::get('/brands/{id}', [BrandController::class, 'show']);

    // Stories Management
    Route::get('/stories', [StoryController::class, 'index']);
    Route::get('/stories/{id}', [StoryController::class, 'show']);
    Route::post('/stories', [StoryController::class, 'store']);
    Route::put('/stories/{id}', [StoryController::class, 'update']);
    Route::delete('/stories/{id}', [StoryController::class, 'destroy']);

    // Testimonials Management
    Route::get('/testimonials', [TestimonialController::class, 'index']);
    Route::get('/testimonials/{id}', [TestimonialController::class, 'show']);
    Route::post('/testimonials', [TestimonialController::class, 'store']);
    Route::put('/testimonials/{id}', [TestimonialController::class, 'update']);
    Route::delete('/testimonials/{id}', [TestimonialController::class, 'destroy']);

    // Pages (Contracts) Management
    Route::get('/pages', [\App\Http\Controllers\Admin\PageController::class, 'index']);
    Route::get('/pages/{id}', [\App\Http\Controllers\Admin\PageController::class, 'show']);
    Route::post('/pages', [\App\Http\Controllers\Admin\PageController::class, 'store']);
    Route::put('/pages/{id}', [\App\Http\Controllers\Admin\PageController::class, 'update']);
    Route::delete('/pages/{id}', [\App\Http\Controllers\Admin\PageController::class, 'destroy']);

    // Bank Accounts Management
    Route::get('/bank-accounts', [\App\Http\Controllers\Admin\BankAccountController::class, 'index']);

    // Settings
    Route::prefix('settings')->group(function () {
        Route::get('/', [SettingsController::class, 'index']);
        Route::get('/{key}', [SettingsController::class, 'show']);
        Route::put('/', [SettingsController::class, 'update']);
        Route::post('/stripe/test', [SettingsController::class, 'testStripe']);
    });

    // Error Logs (Admin only)
    Route::prefix('errors')->group(function () {
        Route::get('/', [ErrorLogController::class, 'index']);
        Route::get('/{id}', [ErrorLogController::class, 'show']);
        Route::post('/{id}/resolve', [ErrorLogController::class, 'resolve']);
        Route::delete('/{id}', [ErrorLogController::class, 'destroy']);
    });

    // Message Management
    Route::apiResource('messages', \App\Http\Controllers\Admin\MessageController::class);

    // Reports
    Route::get('/reports', function (Request $request) {
        $type = $request->get('type', 'all');
        
        if ($type === 'reservations') {
            return app(ReservationManagementController::class)->report($request);
        } elseif ($type === 'payments') {
            return app(PaymentController::class)->report($request);
        } else {
            // Combined report
            $reservationReport = app(ReservationManagementController::class)->report($request);
            $paymentReport = app(PaymentController::class)->report($request);
            
            return response()->json([
                'reservations' => json_decode($reservationReport->getContent(), true),
                'payments' => json_decode($paymentReport->getContent(), true),
            ]);
        }
    });
    Route::prefix('reports')->group(function () {
        Route::get('/reservations', [ReservationManagementController::class, 'report']);
        Route::get('/payments', [PaymentController::class, 'report']);
    });

    // Kasa Yönetimi
    Route::apiResource('cash-desks', CashDeskController::class);
    Route::apiResource('sliders', SliderController::class);
    Route::apiResource('activities', \App\Http\Controllers\Api\Activities\ActivityController::class)->except(['index', 'show']); // Admin specific actions if needed, or full resource
    
    // Visa Management
    Route::apiResource('visas', \App\Http\Controllers\Api\Visas\VisaController::class); // Admin actions

    // Flight Ticket Management
    Route::apiResource('flight-tickets', \App\Http\Controllers\Api\Flights\FlightTicketController::class);
    // Since we used Api\Visas\VisaController for public, we might want a separate Admin controller or reuse.
    // Let's reuse for now but normally we'd separate.
    // Actually, let's just add specific admin routes if needed, or use the resource. 
    // Ideally we want POST/PUT/DELETE protected.
    Route::post('/visas', [\App\Http\Controllers\Api\Visas\VisaController::class, 'store']);
    Route::put('/visas/{id}', [\App\Http\Controllers\Api\Visas\VisaController::class, 'update']);
    Route::delete('/visas/{id}', [\App\Http\Controllers\Api\Visas\VisaController::class, 'destroy']);
    Route::get('/visa-applications', [\App\Http\Controllers\Api\Visas\VisaController::class, 'applications']);

    // Villa Management
    Route::apiResource('villas', VillaController::class);

    // Hotel Management
    Route::apiResource('hotels', HotelController::class);

    // Invoice Management
    Route::prefix('invoices')->group(function () {
        Route::get('/', [InvoiceController::class, 'index']);
        Route::post('/generate', [InvoiceController::class, 'generate']);
        Route::get('/export-csv', [InvoiceController::class, 'exportCsv']);
        Route::get('/{id}/download', [InvoiceController::class, 'download']);
        Route::delete('/{id}', [InvoiceController::class, 'cancel']);
    });
});

// Public API Key Verification (for external systems) - Rate limited
Route::middleware('throttle:30,1')->prefix('verify')->group(function () {
    Route::post('/key', [BrandController::class, 'verifyApiKey']);
    Route::get('/brand', [BrandController::class, 'getByApiKey']);
});

// Webhooks (No Authentication - Signature Verification)
Route::prefix('webhooks')->group(function () {
    Route::post('/stripe', [StripeWebhookController::class, 'handle']);
});

// GetYourGuide (GYG) Integration Endpoints
Route::prefix('gyg')->group(function () {
    Route::get('/products', [\App\Http\Controllers\API\GYG\GYGController::class, 'index']);
    Route::get('/products/{id}', [\App\Http\Controllers\API\GYG\GYGController::class, 'show']);
    Route::get('/availability', [\App\Http\Controllers\API\GYG\GYGController::class, 'availability']);
    Route::post('/bookings', [\App\Http\Controllers\API\GYG\GYGController::class, 'booking']);
});
