<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Activity;
use App\Models\ActivityDate;
use Carbon\Carbon;

class ActivitySeeder extends Seeder
{
    public function run()
    {
        // Truncate existing data to avoid duplicates
        \DB::statement('SET FOREIGN_KEY_CHECKS=0;');
        Activity::truncate();
        ActivityDate::truncate();
        \App\Models\ActivityPricingTier::truncate();
        \DB::statement('SET FOREIGN_KEY_CHECKS=1;');

        // 1. Dubai Desert Safari
        $safari = Activity::create([
            'title' => 'Dubai Çöl Safarisi: 4x4 Dune Bashing, Barbekü ve Deve Turu',
            'slug' => 'dubai-col-safarisi-aksam-yemegi',
            'subtitle' => 'Kızıl kumlarda heyecan dolu bir macera ve Bedevi kampı deneyimi',
            'location' => 'Dubai Çölü, BAE',
            'duration' => '6 Saat',
            'description' => '<p>Dubai\'nin efsanevi kızıl kumlarında 4x4 araçlarla nefes kesen bir safari deneyimine hazır olun. Profesyonel sürücülerimizle kum tepelerinde adrenalin dolu anlar yaşadıktan sonra, gün batımını fotoğraflamak için mola vereceğiz.</p><p>Akşamında geleneksel bir Bedevi kampında barbekü akşam yemeği, oryantal şovlar, nargile ve deve turu ile Arap kültürünü yakından tanıyacaksınız.</p>',
            'base_price' => 55.00,
            'default_start_time' => '15:00',
            'is_active' => true,
            'order' => 1,
            'images' => [
                'https://images.unsplash.com/photo-1451337516015-6b6e9a44a8a3?q=80&w=1000&auto=format&fit=crop', // Desert
                'https://images.unsplash.com/photo-1547234935-80c7142ee969?q=80&w=1000&auto=format&fit=crop', // Camels
                'https://images.unsplash.com/photo-1512453979798-5ea904ac6605?q=80&w=1000&auto=format&fit=crop'  // Sunset
            ],
            'highlights' => ['4x4 Dune Bashing', 'Barbekü Akşam Yemeği', 'Deve Turu & Henna', 'Oryantal ve Ateş Gösterisi'],
            'included' => ['Otelden transfer (4x4)', 'Akşam yemeği (Açık büfe)', 'Sınırsız meşrubat', 'Gösteriler'],
            'excluded' => ['Alkollü içecekler', 'Bahşiş', 'ATV kiralama (Opsiyonel)'],
            'cancellation_policy' => '24 saat öncesine kadar ücretsiz iptal.'
        ]);

        // Pricing Tiers for Safari
        $safari->pricingTiers()->createMany([
            ['name' => 'Yetişkin', 'price' => 55.00, 'min_age' => 12, 'max_age' => null],
            ['name' => 'Çocuk (3-11)', 'price' => 45.00, 'min_age' => 3, 'max_age' => 11],
            ['name' => 'VIP Paket', 'price' => 85.00, 'min_age' => null, 'max_age' => null],
        ]);

        // Dates for Safari
        $startDate = Carbon::now();
        for ($i = 0; $i < 30; $i++) {
             // Skip every Tuesday for maintenance example
            if ($startDate->copy()->addDays($i)->dayOfWeek === Carbon::TUESDAY) continue;

            ActivityDate::create([
                'activity_id' => $safari->id,
                'date' => $startDate->copy()->addDays($i)->format('Y-m-d'),
                'start_time' => '15:00',
                'capacity' => 50,
                'booked' => rand(0, 15),
                'is_active' => true,
                'pricing_data' => [
                    // Dynamic pricing example: weekends slightly more expensive
                    // Assuming tier IDs: 1 (Adult), 2 (Child), 3 (VIP). 
                    // Note: In seeder we can't easily guess IDs without return, but for sim plicity let's rely on base pricing or update logic if strictly needed.
                    // For now, simpler to leave pricing_data empty to use defaults, or we can fetch tiers.
                ]
            ]);
        }

        // 2. Burj Khalifa At the Top
        $burj = Activity::create([
            'title' => 'Burj Khalifa: 124. ve 125. Kat Giriş Bileti',
            'slug' => 'burj-khalifa-at-the-top-giris',
            'subtitle' => 'Dünyanın en yüksek binasından Dubai manzarası',
            'location' => 'Downtown Dubai',
            'duration' => '2 Saat',
            'description' => '<p>Dünyanın en yüksek binası Burj Khalifa\'nın 124. ve 125. katlarına (At the Top) çıkarak Dubai\'yi kuşbakışı izleyin. Dünyanın en hızlı asansörlerinden biriyle sadece 60 saniyede zirveye ulaşacaksınız.</p><p>Panoramik teraslardan şehrin, çölün ve okyanusun eşsiz manzarasının keyfini çıkarın. Özellikle gün batımı saatlerinde manzara büyüleyicidir.</p>',
            'base_price' => 45.00,
            'default_start_time' => '10:00',
            'is_active' => true,
            'order' => 2,
            'images' => [
                'https://images.unsplash.com/photo-1582672060674-bc2bd808a8b5?q=80&w=1000&auto=format&fit=crop', // Burj Khalifa
                'https://images.unsplash.com/photo-1518684079858-191c49ea0765?q=80&w=1000&auto=format&fit=crop', // View
                'https://images.unsplash.com/photo-1526495124232-a04e1849168c?q=80&w=1000&auto=format&fit=crop'  // Fountain
            ],
            'highlights' => ['124. ve 125. Kat Gözlem Terası', 'Dünyanın en hızlı asansörü', '360 derece panoramik manzara', 'Teleskop ile inceleme'],
            'included' => ['Giriş Bileti (Fast Track opsiyonel)', 'Wi-Fi'],
            'excluded' => ['Transfer', 'Yiyecek & İçecek', '148. Kat Erişimi'],
            'cancellation_policy' => 'Etkinlik saatinden 24 saat önce iptal edilebilir.'
        ]);

        $burj->pricingTiers()->createMany([
            ['name' => 'Yetişkin', 'price' => 45.00, 'min_age' => 12, 'max_age' => null],
            ['name' => 'Çocuk (4-11)', 'price' => 35.00, 'min_age' => 4, 'max_age' => 11],
            ['name' => 'Bebek (0-3)', 'price' => 0.00, 'min_age' => 0, 'max_age' => 3],
        ]);

        // Multiple slots per day for Burj Khalifa
        $burjDate = Carbon::tomorrow();
        $times = ['10:00', '12:00', '14:00', '16:00', '18:00', '20:00'];
        
        foreach ($times as $time) {
            ActivityDate::create([
                'activity_id' => $burj->id,
                'date' => $burjDate->format('Y-m-d'),
                'start_time' => $time,
                'capacity' => 100,
                'booked' => rand(20, 90),
                'is_active' => true
            ]);
        }


        // 3. Dubai Marina Yacht Tour
        $marina = Activity::create([
            'title' => 'Dubai Marina: Lüks Yat ile Akşam Yemeği',
            'slug' => 'dubai-marina-yat-turu-aksam-yemegi',
            'subtitle' => 'Işıklar altında Dubai Marina ve Palm Jumeirah manzarası',
            'location' => 'Dubai Marina',
            'duration' => '3 Saat',
            'description' => '<p>Dubai Marina\'nın gökdelenleri ve ışıkları arasında lüks bir yat ile süzülün. Ain Dubai ve Palm Jumeirah manzaraları eşliğinde 5 yıldızlı açık büfe akşam yemeğinin tadını çıkarın.</p><p>Canlı müzik eşliğinde romantik ve keyifli bir akşam sizi bekliyor.</p>',
            'base_price' => 70.00,
            'default_start_time' => '20:00',
            'is_active' => true,
            'order' => 3,
            'images' => [
                'https://images.unsplash.com/photo-1512418490979-92798cec1380?q=80&w=1000&auto=format&fit=crop', // Marina night
                'https://images.unsplash.com/photo-1544620347-c4fd4a3d5957?q=80&w=1000&auto=format&fit=crop', // Boat
                'https://images.unsplash.com/photo-1629239828250-9856bb3923a1?q=80&w=1000&auto=format&fit=crop'  // Ain Dubai
            ],
            'highlights' => ['Lüks Yat Turu', 'Açık Büfe Akşam Yemeği', 'Canlı Müzik', 'Ain Dubai Manzarası'],
            'included' => ['Yat Gezisi', 'Yemek', 'Hoşgeldin içeceği'],
            'excluded' => ['Otel Transferi', 'Özel Fotoğraf Çekimi'],
            'cancellation_policy' => '24 saat öncesine kadar iade.'
        ]);

         $marina->pricingTiers()->createMany([
            ['name' => 'Standart Bilet', 'price' => 70.00, 'min_age' => null, 'max_age' => null],
            ['name' => 'VIP Masa (Cam Kenarı)', 'price' => 95.00, 'min_age' => null, 'max_age' => null],
        ]);

        ActivityDate::create([
            'activity_id' => $marina->id,
            'date' => Carbon::tomorrow()->format('Y-m-d'),
            'start_time' => '20:00',
            'capacity' => 60,
            'booked' => 45, // Popular
            'is_active' => true
        ]);
        
        ActivityDate::create([
            'activity_id' => $marina->id,
            'date' => Carbon::tomorrow()->addDay()->format('Y-m-d'),
            'start_time' => '20:00',
            'capacity' => 60,
            'booked' => 10,
            'is_active' => true
        ]);
    }
}

