<?php

namespace App\Http\Controllers\Webhooks;

use App\Http\Controllers\Controller;
use App\Models\Payment;
use App\Models\Reservation;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Stripe\StripeClient;
use Stripe\Webhook;

class StripeWebhookController extends Controller
{
    private StripeClient $stripe;

    public function __construct()
    {
        $this->stripe = new StripeClient(config('services.stripe.secret'));
    }

    /**
     * Handle Stripe webhook
     */
    public function handle(Request $request): JsonResponse
    {
        $payload = $request->getContent();
        $sigHeader = $request->header('Stripe-Signature');
        $endpointSecret = config('services.stripe.webhook_secret');

        try {
            $event = Webhook::constructEvent($payload, $sigHeader, $endpointSecret);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Invalid signature'
            ], 400);
        }

        switch ($event->type) {
            case 'checkout.session.completed':
                $this->handleCheckoutSessionCompleted($event->data->object);
                break;

            case 'payment_intent.succeeded':
                $this->handlePaymentIntentSucceeded($event->data->object);
                break;

            case 'payment_intent.payment_failed':
                $this->handlePaymentIntentFailed($event->data->object);
                break;

            case 'payment_link.payment_failed':
                $this->handlePaymentLinkFailed($event->data->object);
                break;
        }

        return response()->json(['received' => true]);
    }

    private function handleCheckoutSessionCompleted($session)
    {
        $reservationId = $session->metadata->reservation_id ?? null;
        if (!$reservationId) return;

        $reservation = Reservation::find($reservationId);
        if (!$reservation) return;

        $payment = Payment::where('stripe_payment_intent_id', $session->payment_intent)
            ->first();

        if ($payment) {
            $payment->update([
                'status' => 'completed',
                'paid_at' => now(),
            ]);

            // Update reservation
            $reservation->increment('paid_amount', $payment->amount);
            $reservation->decrement('remaining_amount', $payment->amount);

            if ($reservation->remaining_amount <= 0) {
                $reservation->update([
                    'status' => 'payment_completed',
                ]);
            } else {
                $reservation->update([
                    'status' => 'payment_partial',
                ]);
            }

            // TODO: Generate documents and send email
            // Event: PaymentCompleted
            // Listener: GenerateDocuments, SendDocumentsEmail
        }
    }

    private function handlePaymentIntentSucceeded($intent)
    {
        $payment = Payment::where('stripe_payment_intent_id', $intent->id)->first();
        if (!$payment) return;

        $payment->update([
            'status' => 'completed',
            'paid_at' => now(),
        ]);

        $reservation = $payment->reservation;
        $reservation->increment('paid_amount', $payment->amount);
        $reservation->decrement('remaining_amount', $payment->amount);

        if ($reservation->remaining_amount <= 0) {
            $reservation->update(['status' => 'payment_completed']);
        } else {
            $reservation->update(['status' => 'payment_partial']);
        }
    }

    private function handlePaymentIntentFailed($intent)
    {
        $payment = Payment::where('stripe_payment_intent_id', $intent->id)->first();
        if ($payment) {
            $payment->update(['status' => 'failed']);
        }
    }

    private function handlePaymentLinkFailed($paymentLink)
    {
        // Handle payment link failure
    }
}
