<?php

namespace App\Http\Controllers\Api\Visas;

use App\Http\Controllers\Controller;
use App\Models\Visa;
use App\Models\VisaApplication;
use Illuminate\Http\Request;

class VisaController extends Controller
{
    public function index()
    {
        $visas = Visa::all(); // Admin panel usually wants all visas
        return response()->json(['data' => $visas]);
    }

    public function show($id)
    {
        if (is_numeric($id)) {
            $visa = Visa::findOrFail($id);
        } else {
            $visa = Visa::where('slug', $id)->firstOrFail();
        }
        return response()->json(['data' => $visa]);
    }

    public function storeApplication(Request $request)
    {
        $validated = $request->validate([
            'visa_id' => 'required|exists:visas,id',
            'contact_info' => 'required|array',
            'contact_info.name' => 'required|string',
            'contact_info.surname' => 'required|string',
            'contact_info.email' => 'required|email',
            'contact_info.phone' => 'required|string',
            'documents' => 'required|array',
            'documents.passport' => 'required|string', // URL from upload
            'documents.photo' => 'required|string'    // URL from upload
        ]);

        $application = VisaApplication::create([
            'visa_id' => $validated['visa_id'],
            'contact_info' => $validated['contact_info'],
            'documents' => $validated['documents'],
            'status' => 'pending',
            'payment_status' => 'pending'
        ]);

        return response()->json([
            'message' => 'Visa application submitted successfully.',
            'application_id' => $application->id
        ], 201);
    }

    // Admin Methods
    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string',
            'slug' => 'required|string|unique:visas',
            'price' => 'required|numeric',
            'description' => 'nullable|string',
            'processing_time' => 'nullable|string',
            'requirements' => 'nullable|array',
            'is_active' => 'boolean'
        ]);

        $visa = Visa::create($validated);
        return response()->json($visa, 201);
    }

    public function update(Request $request, $id)
    {
        $visa = Visa::findOrFail($id);
        $validated = $request->validate([
            'title' => 'string',
            'slug' => 'string|unique:visas,slug,' . $id,
            'price' => 'numeric',
            'description' => 'nullable|string',
            'processing_time' => 'nullable|string',
            'requirements' => 'nullable|array',
            'is_active' => 'boolean'
        ]);

        $visa->update($validated);
        return response()->json($visa);
    }

    public function destroy($id)
    {
        Visa::findOrFail($id)->delete();
        return response()->noContent();
    }

    public function applications()
    {
        $applications = VisaApplication::with('visa')->latest()->get();
        return response()->json($applications);
    }
}
