<?php

namespace App\Http\Controllers\API\Tours;

use App\Http\Controllers\Controller;
use App\Models\Reservation;
use App\Models\Tour;
use App\Models\TourDate;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;

class TourController extends Controller
{
    /**
     * Get all tours
     */
    public function index(Request $request): JsonResponse
    {
        $query = Tour::with(['brand', 'dates', 'categories'])
            ->where('is_active', true);

        if ($request->has('brand_id')) {
            $query->where('brand_id', $request->brand_id);
        }

        if ($request->has('type')) {
            $query->where('type', $request->type);
        }

        if ($request->has('category_id')) {
            $query->whereHas('categories', function ($q) use ($request) {
                $q->where('tour_categories.id', $request->category_id);
            });
        }

        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                  ->orWhere('subtitle', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }

        $tours = $query->orderBy('order')
            ->orderBy('created_at', 'desc')
            ->paginate($request->get('per_page', 15));

        return response()->json($tours);
    }

    /**
     * Get tour by ID
     */
    public function show(int $id): JsonResponse
    {
        try {
            $tour = Tour::with(['brand', 'dates', 'categories'])
                ->findOrFail($id);
            
            $tourArray = $tour->toArray();
            
            // departure_point'ı direkt DB'den oku (cast bypass) ve doğru formatta döndür
            $departurePointRaw = DB::table('tours')
                ->where('id', $id)
                ->value('departure_point');
            
            // Log for debugging
            \Log::info('Departure point raw from DB', [
                'id' => $id,
                'raw' => $departurePointRaw,
                'type' => gettype($departurePointRaw)
            ]);
            
            if ($departurePointRaw !== null && $departurePointRaw !== '') {
                // MySQL JSON column returns as string if it's JSON, or as is
                if (is_string($departurePointRaw)) {
                    // Check if it's already a JSON string
                    $decoded = json_decode($departurePointRaw, true);
                    if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                        // It's valid JSON, re-encode to ensure consistency
                        $tourArray['departure_point'] = json_encode($decoded);
                    } else {
                        // It's a plain string, use as is
                        $tourArray['departure_point'] = $departurePointRaw;
                    }
                } else if (is_array($departurePointRaw)) {
                    // Already decoded, encode it
                    $tourArray['departure_point'] = json_encode($departurePointRaw);
                } else {
                    $tourArray['departure_point'] = null;
                }
            } else {
                $tourArray['departure_point'] = null;
            }
            
            \Log::info('Departure point final', [
                'final' => $tourArray['departure_point'],
                'type' => gettype($tourArray['departure_point'])
            ]);
            
            // Diğer JSON alanlarını string olarak döndür
            $jsonFields = ['description', 'itinerary', 'images', 'included', 'excluded', 'highlights', 'room_options', 'hotel_options', 'feature_cards', 'countries', 'flight_options', 'label', 'seo_title'];
            foreach ($jsonFields as $field) {
                if (isset($tourArray[$field])) {
                    if (is_array($tourArray[$field])) {
                        // Boş array ise null döndür, değilse JSON string'e çevir
                        if (empty($tourArray[$field])) {
                            $tourArray[$field] = null;
                        } else {
                            $tourArray[$field] = json_encode($tourArray[$field]);
                        }
                    } elseif ($tourArray[$field] === null) {
                        // Zaten null ise olduğu gibi bırak
                        $tourArray[$field] = null;
                    }
                    // String ise olduğu gibi bırak (zaten JSON string olabilir)
                }
            }

            // pickup_points'i özel olarak işle
            $pickupPointsRaw = $tour->pickup_points; // Model cast nedeniyle array gelebilir
            if (!empty($pickupPointsRaw)) {
                 if (is_array($pickupPointsRaw)) {
                     $tourArray['pickup_points'] = json_encode($pickupPointsRaw);
                 } elseif (is_string($pickupPointsRaw)) {
                     // Zaten string ise (örn: "[...]")
                     $tourArray['pickup_points'] = $pickupPointsRaw;
                 }
            } else {
                // Eğer modelden boş geldiyse, DB'ye direkt bak (cast hatası ihtimaline karşı)
                 $rawPickupDB = DB::table('tours')->where('id', $id)->value('pickup_points');
                 if ($rawPickupDB) {
                     $tourArray['pickup_points'] = $rawPickupDB; // String olarak gelir
                 } else {
                     $tourArray['pickup_points'] = null;
                 }
            }
            
            // Dates ilişkisini manuel olarak dönüştür - single_supplement_price dahil et
            if (isset($tourArray['dates']) && is_array($tourArray['dates']) && $tour->dates) {
                $tourArray['dates'] = $tour->dates->map(function ($dateModel) {
                    $dateArray = [
                        'id' => $dateModel->id,
                        'tour_id' => $dateModel->tour_id,
                        'start_date' => $dateModel->start_date ? $dateModel->start_date->format('Y-m-d\TH:i:s.u\Z') : null,
                        'end_date' => $dateModel->end_date ? $dateModel->end_date->format('Y-m-d\TH:i:s.u\Z') : null,
                        'start_time' => $dateModel->start_time ? $dateModel->start_time->format('H:i:s') : null,
                        'end_time' => $dateModel->end_time ? $dateModel->end_time->format('H:i:s') : null,
                        'adult_price' => $dateModel->adult_price ? (string)$dateModel->adult_price : null,
                        'adult_discounted_price' => $dateModel->adult_discounted_price ? (string)$dateModel->adult_discounted_price : null,
                        'child_price' => $dateModel->child_price ? (string)$dateModel->child_price : null,
                        'child_discounted_price' => $dateModel->child_discounted_price ? (string)$dateModel->child_discounted_price : null,
                        'baby_price' => $dateModel->baby_price ? (string)$dateModel->baby_price : null,
                        'baby_discounted_price' => $dateModel->baby_discounted_price ? (string)$dateModel->baby_discounted_price : null,
                        'capacity' => $dateModel->capacity,
                        'booked' => $dateModel->booked,
                        'status' => $dateModel->status,
                        'pickup_point_prices' => $dateModel->pickup_point_prices,
                        'created_at' => $dateModel->created_at ? $dateModel->created_at->toISOString() : null,
                        'updated_at' => $dateModel->updated_at ? $dateModel->updated_at->toISOString() : null,
                    ];
                    
                    // single_supplement_price: DB'den direkt oku (cast bypass)
                    $dbValue = DB::table('tour_dates')
                        ->where('id', $dateModel->id)
                        ->value('single_supplement_price');
                    
                    // Değeri string olarak ekle (null ise null, 0 ise '0', diğerleri string)
                    if ($dbValue !== null && $dbValue !== '') {
                        $dateArray['single_supplement_price'] = (string)$dbValue;
                    } else if ($dbValue === 0 || $dbValue === 0.0) {
                        $dateArray['single_supplement_price'] = '0';
                    } else {
                        $dateArray['single_supplement_price'] = null;
                    }
                    
                    $dateArray['transportation'] = $dateModel->transportation;
                    $dateArray['hotel_options'] = $dateModel->hotel_options;
                    $dateArray['bus_layout'] = $dateModel->bus_layout;
                    
                    return $dateArray;
                })->toArray();
            }
            
            // SEO alanlarının response'da olduğundan emin ol
            if (!isset($tourArray['slug'])) $tourArray['slug'] = null;
            if (!isset($tourArray['seo_description'])) $tourArray['seo_description'] = null;
            if (!isset($tourArray['seo_image'])) $tourArray['seo_image'] = null;
            if (!isset($tourArray['seo_title'])) $tourArray['seo_title'] = null;
            if (!isset($tourArray['subtitle'])) $tourArray['subtitle'] = null;
            if (!isset($tourArray['label'])) $tourArray['label'] = null;

            // ---------------------------------------------------------
            // 1. TERMS (Sözleşmeler) - Page modelinden çek
            // ---------------------------------------------------------
            $termsList = [];
            $termsIdsRaw = $tour->terms_ids; // Model cast ettiği için array gelebilir veya null
            
            $termsIds = $termsIdsRaw;
            if (is_string($termsIdsRaw)) {
                $termsIds = json_decode($termsIdsRaw, true);
            }
            
            if (!empty($termsIds) && is_array($termsIds)) {
                $pages = \App\Models\Page::whereIn('id', $termsIds)->get();
                
                foreach ($termsIds as $pageId) {
                    $page = $pages->firstWhere('id', $pageId);
                    if ($page) {
                        $pageContent = '';
                        if (!empty($page->sections) && is_array($page->sections)) {
                            foreach ($page->sections as $section) {
                                if (isset($section['type']) && $section['type'] === 'content' && isset($section['data']['html'])) {
                                    $pageContent .= $section['data']['html'] . "<br><br>";
                                }
                            }
                        }
                        
                        // Başlık ve içeriği yapılandırılmış olarak ekle
                        $termsList[] = [
                            'id' => $page->id,
                            'title' => $page->title, // Frontend başlığı gösterecek
                            'content' => $pageContent // Frontend HTML'i render edecek
                        ];
                    }
                }
            }
            
            $tourArray['terms'] = !empty($termsList) ? $termsList : null;

            // ---------------------------------------------------------
            // 2. BANK ACCOUNT - BankAccount modelinden çek
            // ---------------------------------------------------------
            $bankAccounts = [];
            $bankAccountIds = [];

            // A. Yeni çoklu seçim sütununu kontrol et
            if (!empty($tour->bank_account_ids)) {
                // Eğer string gelirse (cast çalışmazsa) decode et
                if (is_string($tour->bank_account_ids)) {
                    $decoded = json_decode($tour->bank_account_ids, true);
                    if (is_array($decoded)) {
                        $bankAccountIds = $decoded;
                    }
                } elseif (is_array($tour->bank_account_ids)) {
                    $bankAccountIds = $tour->bank_account_ids;
                }
            }
            // B. Eski tekli seçim sütununu kontrol et (fallback)
            elseif ($tour->bank_account_id) {
                $bankAccountIds = [$tour->bank_account_id];
            }

            if (!empty($bankAccountIds)) {
                $accounts = \App\Models\BankAccount::whereIn('id', $bankAccountIds)
                    ->where('is_active', true)
                    ->get();

                foreach ($accounts as $bankAccount) {
                    $bankAccounts[] = [
                        'bank' => $bankAccount->bank_name,
                        'currency' => 'EUR', // Varsayılan veya modelde varsa oradan
                        'holder' => $bankAccount->account_holder,
                        'iban' => $bankAccount->iban,
                        'account_number' => $bankAccount->account_number,
                        'bic' => $bankAccount->swift, // Swift/BIC
                    ];
                }
            }
            
            $tourArray['bank_accounts'] = $bankAccounts;

            return response()->json([
                'data' => $tourArray
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get tour dates
     */
    public function dates(int $id): JsonResponse
    {
        $tour = Tour::findOrFail($id);
        
        $dates = TourDate::where('tour_id', $id)
            ->where('start_date', '>=', now()->toDateString())
            ->orderBy('start_date')
            ->get()
            ->map(function ($dateModel) {
                $dateArray = [
                    'id' => $dateModel->id,
                    'tour_id' => $dateModel->tour_id,
                    'start_date' => $dateModel->start_date ? $dateModel->start_date->format('Y-m-d\TH:i:s.u\Z') : null,
                    'end_date' => $dateModel->end_date ? $dateModel->end_date->format('Y-m-d\TH:i:s.u\Z') : null,
                    'start_time' => $dateModel->start_time ? $dateModel->start_time->format('H:i:s') : null,
                    'end_time' => $dateModel->end_time ? $dateModel->end_time->format('H:i:s') : null,
                    'adult_price' => $dateModel->adult_price ? (string)$dateModel->adult_price : null,
                    'adult_discounted_price' => $dateModel->adult_discounted_price ? (string)$dateModel->adult_discounted_price : null,
                    'child_price' => $dateModel->child_price ? (string)$dateModel->child_price : null,
                    'child_discounted_price' => $dateModel->child_discounted_price ? (string)$dateModel->child_discounted_price : null,
                    'baby_price' => $dateModel->baby_price ? (string)$dateModel->baby_price : null,
                    'baby_discounted_price' => $dateModel->baby_discounted_price ? (string)$dateModel->baby_discounted_price : null,
                    'capacity' => $dateModel->capacity,
                    'booked' => $dateModel->booked,
                    'status' => $dateModel->status,
                    'pickup_point_prices' => $dateModel->pickup_point_prices,
                ];
                
                // single_supplement_price: DB'den direkt oku (cast bypass)
                $dbValue = DB::table('tour_dates')
                    ->where('id', $dateModel->id)
                    ->value('single_supplement_price');
                
                // Değeri string olarak ekle
                if ($dbValue !== null && $dbValue !== '') {
                    $dateArray['single_supplement_price'] = (string)$dbValue;
                } else if ($dbValue === 0 || $dbValue === 0.0) {
                    $dateArray['single_supplement_price'] = '0';
                } else {
                    $dateArray['single_supplement_price'] = null;
                }
                
                $dateArray['transportation'] = $dateModel->transportation;
                $dateArray['hotel_options'] = $dateModel->hotel_options;
                $dateArray['bus_layout'] = $dateModel->bus_layout;
                
                return $dateArray;
            });

        return response()->json([
            'dates' => $dates->values()
        ]);
    }

    /**
     * Get tour availability
     */
    public function availability(Request $request, int $id): JsonResponse
    {
        $tour = Tour::findOrFail($id);
        $dateId = $request->get('date_id');
        $guests = $request->get('guests', []);

        if (!$dateId) {
            return response()->json([
                'message' => 'Date ID is required'
            ], 422);
        }

        $tourDate = TourDate::where('tour_id', $id)
            ->where('id', $dateId)
            ->firstOrFail();

        $totalGuests = array_sum(array_column($guests, 'count'));
        $available = $tourDate->available_spots >= $totalGuests;

        return response()->json([
            'available' => $available,
            'available_spots' => $tourDate->available_spots,
            'requested_spots' => $totalGuests,
            'tour_date' => $tourDate,
        ]);
    }

    /**
     * Get tour pricing
     */
    public function pricing(Request $request, int $id): JsonResponse
    {
        $tour = Tour::findOrFail($id);
        $dateId = $request->get('date_id');
        $guests = $request->get('guests', []);
        $pickupPointId = $request->get('pickup_point_id');

        if (!$dateId) {
            return response()->json([
                'message' => 'Date ID is required'
            ], 422);
        }

        $tourDate = TourDate::where('tour_id', $id)
            ->where('id', $dateId)
            ->firstOrFail();

        $total = 0;
        $breakdown = [];

        foreach ($guests as $guest) {
            $type = $guest['type'];
            $count = $guest['count'] ?? 1;

            $price = match($type) {
                'adult' => $tourDate->adult_discounted_price ?? $tourDate->adult_price,
                'student' => $tourDate->adult_discounted_price ?? $tourDate->adult_price,
                'child' => $tourDate->child_discounted_price ?? $tourDate->child_price ?? 0,
                'baby' => $tourDate->baby_discounted_price ?? $tourDate->baby_price ?? 0,
                default => 0,
            };

            $subtotal = $price * $count;
            $total += $subtotal;

            $breakdown[] = [
                'type' => $type,
                'count' => $count,
                'unit_price' => $price,
                'subtotal' => $subtotal,
            ];
        }

        // Pickup point price modifier
        if ($pickupPointId) {
            $pickupPoint = \App\Models\PickupPoint::find($pickupPointId);
            if ($pickupPoint && $pickupPoint->price_modifier != 0) {
                $modifier = $pickupPoint->price_modifier;
                $total += $modifier;
                $breakdown[] = [
                    'type' => 'pickup_point',
                    'description' => $pickupPoint->location,
                    'modifier' => $modifier,
                ];
            }
        }

        return response()->json([
            'total' => $total,
            'breakdown' => $breakdown,
            'tour_date' => [
                'id' => $tourDate->id,
                'start_date' => $tourDate->start_date->format('Y-m-d'),
            ],
        ]);
    }

    /**
     * Create tour (Admin only)
     */
    public function store(Request $request): JsonResponse
    {
        return response()->json(['message' => 'Not implemented'], 501);
    }

    /**
     * Update tour (Admin only)
     */
    public function update(Request $request, int $id): JsonResponse
    {
        return response()->json(['message' => 'Not implemented'], 501);
    }

    /**
     * Delete tour (Admin only)
     */
    public function destroy(int $id): JsonResponse
    {
        return response()->json(['message' => 'Not implemented'], 501);
    }

    /**
     * Get occupied seat numbers for a tour date (from all non-cancelled reservations).
     * Used by the reservation wizard so already-sold seats are shown as unavailable.
     */
    public function occupiedSeats(Request $request): JsonResponse
    {
        $tourDateId = $request->query('tour_date_id');
        if (!$tourDateId) {
            return response()->json(['message' => 'tour_date_id is required'], 422);
        }

        $numbers = Reservation::query()
            ->where('tour_date_id', (int) $tourDateId)
            ->whereNotIn('status', ['cancelled'])
            ->with('guests')
            ->get()
            ->flatMap(function (Reservation $r) {
                return $r->guests->flatMap(function ($g) {
                    $seats = [];
                    if ($g->seat_number !== null && $g->seat_number !== '') {
                        $num = (int) $g->seat_number;
                        if ($num >= 1) {
                            $seats[] = $num;
                        }
                    }
                    if ($g->extra_seat_number !== null && $g->extra_seat_number !== '') {
                        $num = (int) $g->extra_seat_number;
                        if ($num >= 1) {
                            $seats[] = $num;
                        }
                    }
                    return $seats;
                });
            })
            ->unique()
            ->values()
            ->sort()
            ->values()
            ->toArray();

        return response()->json([
            'occupied_seat_numbers' => $numbers,
        ]);
    }
}
