<?php

namespace App\Http\Controllers\API\Payments;

use App\Http\Controllers\Controller;
use App\Models\Payment;
use App\Models\Reservation;
use App\Models\PaymentLink;
use App\Models\Tour;
use App\Models\TourDate;
use App\Models\Room;
use App\Models\Guest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Schema;
use Illuminate\Http\JsonResponse;
use Stripe\StripeClient;
use Illuminate\Support\Str;

class PaymentController extends Controller
{
    private ?StripeClient $stripe = null;

    private function getStripe(): StripeClient
    {
        if ($this->stripe === null) {
            $key = config('services.stripe.secret');
            if (empty($key)) {
                throw new \RuntimeException('Stripe API anahtarı yapılandırılmamış. Ön rezervasyon için Stripe gerekmez; ödeme linki operatör onayı sonrası e-posta ile gönderilir.');
            }
            $this->stripe = new StripeClient($key);
        }
        return $this->stripe;
    }

    /**
     * Create Stripe Payment Intent for Elements
     */
    public function createPaymentIntent(Request $request): JsonResponse
    {
        $validator = \Illuminate\Support\Facades\Validator::make($request->all(), [
            'reservation_id' => 'required|exists:reservations,id',
            'amount' => 'required|numeric|min:0.01',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $reservation = Reservation::findOrFail($request->reservation_id);

        if ($reservation->user_id !== $request->user()->id && !$request->user()->isAdmin()) {
            return response()->json([
                'message' => 'Unauthorized'
            ], 403);
        }

        try {
            // Create Payment Intent
            $paymentIntent = $this->getStripe()->paymentIntents->create([
                'amount' => (int)($request->amount * 100), // Convert to cents
                'currency' => 'eur',
                'payment_method_types' => ['card'],
                'metadata' => [
                    'reservation_id' => $reservation->id,
                    'user_id' => $request->user()->id,
                    'reservation_code' => $reservation->reservation_code,
                ],
            ]);

            // Create payment record
            $payment = Payment::create([
                'reservation_id' => $reservation->id,
                'user_id' => $request->user()->id,
                'payment_method' => 'stripe',
                'amount' => $request->amount,
                'status' => 'pending',
                'stripe_payment_intent_id' => $paymentIntent->id,
            ]);

            return response()->json([
                'client_secret' => $paymentIntent->client_secret,
                'payment_intent_id' => $paymentIntent->id,
                'payment_id' => $payment->id,
            ]);

        } catch (\Exception $e) {
            \Log::error('Payment Intent creation failed', [
                'error' => $e->getMessage(),
                'reservation_id' => $request->reservation_id,
            ]);

            return response()->json([
                'message' => 'Payment Intent creation failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Confirm payment intent
     */
    public function confirmPayment(Request $request): JsonResponse
    {
        $validator = \Illuminate\Support\Facades\Validator::make($request->all(), [
            'payment_intent_id' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $paymentIntent = $this->getStripe()->paymentIntents->retrieve($request->payment_intent_id);
            
            $payment = Payment::where('stripe_payment_intent_id', $paymentIntent->id)->first();
            
            if (!$payment) {
                return response()->json([
                    'message' => 'Payment not found'
                ], 404);
            }

            return response()->json([
                'status' => $paymentIntent->status,
                'payment' => [
                    'id' => $payment->id,
                    'status' => $payment->status,
                    'amount' => $payment->amount,
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Payment confirmation failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get Stripe public key
     */
    public function getPublicKey(): JsonResponse
    {
        return response()->json([
            'public_key' => config('services.stripe.key'),
        ]);
    }

    /**
     * Pre-reservation from cart (public). Creates reservations only, no payment. Operator approves later, then payment link or havale.
     */
    public function preReservation(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'items' => 'required|array|min:1',
            'items.*.tour_id' => 'required|exists:tours,id',
            'items.*.tour_date_id' => 'required|exists:tour_dates,id',
            'items.*.amount' => 'required|numeric|min:0.01',
            'contact_info' => 'required|array',
            'contact_info.fullName' => 'required|string',
            'contact_info.email' => 'required|email',
            'contact_info.phone' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $contact = $request->contact_info;
        $nameParts = preg_split('/\s+/', trim($contact['fullName'] ?? ''), 2);
        $firstName = $nameParts[0] ?? 'Misafir';
        $lastName = $nameParts[1] ?? '';

        try {
            DB::beginTransaction();
            $reservationIds = [];
            $totalAmount = 0;
            $firstEasyCode = null;

            foreach ($request->items as $item) {
                $tour = Tour::findOrFail($item['tour_id']);
                $amount = (float) $item['amount'];
                $totalAmount += $amount;

                $code = Reservation::generateUnifiedCode($tour->brand_id);
                $reservationCode = $code;
                $easyCode = $code;
                if ($firstEasyCode === null) {
                    $firstEasyCode = $easyCode;
                }

                $cancellationPolicy = in_array($item['cancellation_policy'] ?? null, ['none', 'partial', 'full'])
                    ? $item['cancellation_policy']
                    : 'none';

                $itemFlight = $item['flight'] ?? null;
                $flightPreference = (is_array($itemFlight) && isset($itemFlight['preference'])) ? $itemFlight['preference'] : null;
                $flightType = (is_array($itemFlight) && isset($itemFlight['type'])) ? $itemFlight['type'] : null;
                $airline = (is_array($itemFlight) && isset($itemFlight['airline'])) ? $itemFlight['airline'] : null;
                $customFlightNotes = (is_array($itemFlight) && isset($itemFlight['custom_notes'])) ? $itemFlight['custom_notes'] : null;
                $merchandise = (is_array($item['merchandise'] ?? null) && count($item['merchandise']) > 0) ? $item['merchandise'] : null;

                // Build reservation data
                $reservationData = [
                    'user_id' => $request->user()?->id,
                    'brand_id' => $tour->brand_id,
                    'tour_id' => $tour->id,
                    'tour_date_id' => $item['tour_date_id'],
                    'reservation_code' => $reservationCode,
                    'easy_code' => $easyCode,
                    'status' => 'pending',
                    'total_price' => $amount,
                    'paid_amount' => 0,
                    'remaining_amount' => $amount,
                    'discount_amount' => 0,
                    'payment_type' => 'full',
                    'cancellation_policy' => $cancellationPolicy,
                    'flight_preference' => $flightPreference,
                    'flight_type' => $flightType,
                    'airline' => $airline,
                    'custom_flight_notes' => $customFlightNotes,
                    'contact_info' => array_merge($contact, [
                        'address' => $contact['address'] ?? null,
                        'city' => $contact['city'] ?? null,
                        'zipCode' => $contact['zipCode'] ?? null,
                        'country' => $contact['country'] ?? null,
                        'invoiceType' => $contact['invoiceType'] ?? 'individual',
                        'companyName' => $contact['companyName'] ?? null,
                        'taxID' => $contact['taxID'] ?? null,
                    ]),
                    'channel' => 'website',
                ];

                // Create reservation first (without merchandise in fillable)
                $reservation = Reservation::create($reservationData);
                
                // If merchandise column exists and we have merchandise data, update it via DB
                if (Schema::hasColumn('reservations', 'merchandise') && $merchandise !== null) {
                    DB::table('reservations')
                        ->where('id', $reservation->id)
                        ->update(['merchandise' => json_encode($merchandise)]);
                }

                $itemRooms = $item['rooms'] ?? null;
                $itemGuests = $item['guests'] ?? null;

                // Geçerli hotel_type: rooms tablosundaki ENUM ile uyumlu (sayı/timestamp gönderilirse kabul etme)
                $validHotelTypes = ['3-star', '4-star', '5-star', 'boutique', 'villa'];
                if (is_array($itemRooms) && count($itemRooms) > 0 && is_array($itemGuests) && count($itemGuests) > 0) {
                    $createdRooms = [];
                    foreach ($itemRooms as $r) {
                        $hotelTypeRaw = $r['hotel_type'] ?? $r['hotelType'] ?? null;
                        $hotelType = ($hotelTypeRaw !== null && in_array((string) $hotelTypeRaw, $validHotelTypes, true))
                            ? (string) $hotelTypeRaw
                            : null;
                        $room = Room::create([
                            'reservation_id' => $reservation->id,
                            'room_type' => $r['room_type'] ?? $r['roomType'] ?? 'double',
                            'adults' => (int) ($r['adults'] ?? 0),
                            'students' => (int) ($r['students'] ?? 0),
                            'children' => (int) ($r['children'] ?? 0),
                            'babies' => (int) ($r['babies'] ?? 0),
                            'hotel_type' => $hotelType,
                        ]);
                        $createdRooms[] = $room;
                    }
                    foreach ($itemGuests as $g) {
                        $roomIndex = (int) ($g['room_index'] ?? 0);
                        $roomId = isset($createdRooms[$roomIndex]) ? $createdRooms[$roomIndex]->id : $createdRooms[0]->id;
                        $selectedExtras = $g['selected_extras'] ?? null;
                        if (is_array($selectedExtras)) {
                            $selectedExtras = $selectedExtras;
                        } else {
                            $selectedExtras = null;
                        }
                        Guest::create([
                            'reservation_id' => $reservation->id,
                            'room_id' => $roomId,
                            'type' => $g['type'] ?? 'adult',
                            'first_name' => $g['first_name'] ?? '',
                            'last_name' => $g['last_name'] ?? '',
                            'gender' => $g['gender'] ?? null,
                            'date_of_birth' => !empty($g['date_of_birth']) ? $g['date_of_birth'] : null,
                            'citizenship' => $g['citizenship'] ?? null,
                            'pickup_point_id' => (int) ($g['pickup_point_id'] ?? 0) ?: null,
                            'seat_number' => $g['seat_number'] ?? null,
                            'extra_seat_number' => $g['extra_seat_number'] ?? null,
                            'selected_extras' => $selectedExtras,
                            'notes' => $g['notes'] ?? null,
                        ]);
                    }
                } else {
                    $room = Room::create([
                        'reservation_id' => $reservation->id,
                        'room_type' => 'double',
                        'adults' => 1,
                        'students' => 0,
                        'children' => 0,
                        'babies' => 0,
                    ]);
                    Guest::create([
                        'reservation_id' => $reservation->id,
                        'room_id' => $room->id,
                        'type' => 'adult',
                        'first_name' => $firstName,
                        'last_name' => $lastName,
                    ]);
                }

                $reservationIds[] = $reservation->id;
            }

            DB::commit();

            return response()->json([
                'reservation_ids' => $reservationIds,
                'easy_code' => $firstEasyCode,
                'total_amount' => $totalAmount,
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            
            // Always log to file first (most reliable)
            \Log::error('Pre-reservation failed', [
                'error' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString(),
                'request_data' => [
                    'items_count' => count($request->items ?? []),
                    'endpoint' => $request->fullUrl(),
                ]
            ]);
            
            // Reconnect so ErrorLog insert uses a clean connection (avoids failed state after DB error)
            try {
                DB::reconnect();
            } catch (\Throwable $reconnectEx) {
                \Log::warning('DB reconnect after pre-reservation error failed', ['msg' => $reconnectEx->getMessage()]);
            }
            
            // Try to log to database (try-catch to prevent error logging from failing)
            $errorLogId = null;
            try {
                $errorLog = \App\Services\ErrorLogService::log(
                    $e,
                    $request,
                    'Backend',
                    500,
                    [
                        'action' => 'pre_reservation',
                        'items_count' => count($request->items ?? []),
                    ]
                );
                $errorLogId = $errorLog?->id ?? null;
                
                if ($errorLogId) {
                    \Log::info('Error logged to database', ['error_log_id' => $errorLogId]);
                }
            } catch (\Exception $logError) {
                // If error logging fails, just log to file (already done above)
                \Log::warning('Failed to log error to database', [
                    'original_error' => $e->getMessage(),
                    'logging_error' => $logError->getMessage(),
                ]);
            }
            
            // User-friendly error message (don't expose technical details)
            $userMessage = 'Ön rezervasyon oluşturulurken bir hata oluştu. Lütfen tekrar deneyin veya destek ekibiyle iletişime geçin.';
            
            // Check for specific database errors
            if (str_contains($e->getMessage(), 'Base table or view not found')) {
                $userMessage = 'Sistem yapılandırması eksik. Lütfen yöneticiye bildirin.';
            } elseif (str_contains($e->getMessage(), 'Column not found')) {
                $userMessage = 'Veritabanı güncellemesi gerekli. Lütfen yöneticiye bildirin.';
            } elseif (str_contains($e->getMessage(), 'SQLSTATE')) {
                $userMessage = 'Veritabanı hatası oluştu. Lütfen tekrar deneyin.';
            }
            
            return response()->json([
                'message' => $userMessage,
                'error_log_id' => $errorLogId,
            ], 500);
        }
    }

    /**
     * Checkout from cart (public – no auth). Creates reservations + PaymentIntent, returns client_secret.
     */
    public function fromCart(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'items' => 'required|array|min:1',
            'items.*.tour_id' => 'required|exists:tours,id',
            'items.*.tour_date_id' => 'required|exists:tour_dates,id',
            'items.*.amount' => 'required|numeric|min:0.01',
            'contact_info' => 'required|array',
            'contact_info.fullName' => 'required|string',
            'contact_info.email' => 'required|email',
            'contact_info.phone' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $contact = $request->contact_info;
        $nameParts = preg_split('/\s+/', trim($contact['fullName'] ?? ''), 2);
        $firstName = $nameParts[0] ?? 'Misafir';
        $lastName = $nameParts[1] ?? '';

        try {
            DB::beginTransaction();
            $reservationIds = [];
            $totalAmount = 0;

            foreach ($request->items as $item) {
                $tour = Tour::findOrFail($item['tour_id']);
                $amount = (float) $item['amount'];
                $totalAmount += $amount;

                $code = Reservation::generateUnifiedCode($tour->brand_id);
                $reservationCode = $code;
                $easyCode = $code;

                $reservation = Reservation::create([
                    'user_id' => $request->user()?->id,
                    'brand_id' => $tour->brand_id,
                    'tour_id' => $tour->id,
                    'tour_date_id' => $item['tour_date_id'],
                    'reservation_code' => $reservationCode,
                    'easy_code' => $easyCode,
                    'status' => 'pending',
                    'total_price' => $amount,
                    'paid_amount' => 0,
                    'remaining_amount' => $amount,
                    'discount_amount' => 0,
                    'payment_type' => 'full',
                    'cancellation_policy' => 'full',
                    'contact_info' => array_merge($contact, [
                        'address' => $contact['address'] ?? null,
                        'city' => $contact['city'] ?? null,
                        'zipCode' => $contact['zipCode'] ?? null,
                        'country' => $contact['country'] ?? null,
                        'invoiceType' => $contact['invoiceType'] ?? 'individual',
                        'companyName' => $contact['companyName'] ?? null,
                        'taxID' => $contact['taxID'] ?? null,
                    ]),
                    'channel' => 'website',
                ]);

                $room = Room::create([
                    'reservation_id' => $reservation->id,
                    'room_type' => 'double',
                    'adults' => 1,
                    'students' => 0,
                    'children' => 0,
                    'babies' => 0,
                ]);

                Guest::create([
                    'reservation_id' => $reservation->id,
                    'room_id' => $room->id,
                    'type' => 'adult',
                    'first_name' => $firstName,
                    'last_name' => $lastName,
                ]);

                $reservationIds[] = $reservation->id;
            }

            $paymentIntent = $this->getStripe()->paymentIntents->create([
                'amount' => (int) round($totalAmount * 100),
                'currency' => 'eur',
                'payment_method_types' => ['card'],
                'metadata' => [
                    'reservation_ids' => implode(',', $reservationIds),
                    'source' => 'from_cart',
                ],
            ]);

            foreach ($reservationIds as $i => $rid) {
                $itemAmount = (float) $request->items[$i]['amount'];
                Payment::create([
                    'reservation_id' => $rid,
                    'user_id' => $request->user()?->id,
                    'payment_method' => 'stripe',
                    'amount' => $itemAmount,
                    'status' => 'pending',
                    'stripe_payment_intent_id' => $paymentIntent->id,
                ]);
            }

            DB::commit();

            return response()->json([
                'client_secret' => $paymentIntent->client_secret,
                'payment_intent_id' => $paymentIntent->id,
                'reservation_ids' => $reservationIds,
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('From-cart checkout failed', ['error' => $e->getMessage()]);
            return response()->json([
                'message' => 'Sipariş oluşturulurken hata oluştu',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Confirm from-cart payment (public). Marks all payments with this intent as completed and updates reservations.
     */
    public function confirmFromCart(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'payment_intent_id' => 'required|string',
        ]);
        if ($validator->fails()) {
            return response()->json(['message' => 'Validation error', 'errors' => $validator->errors()], 422);
        }
        try {
            $intent = $this->getStripe()->paymentIntents->retrieve($request->payment_intent_id);
            if ($intent->status !== 'succeeded') {
                return response()->json(['message' => 'Ödeme henüz tamamlanmadı'], 422);
            }
            $payments = Payment::where('stripe_payment_intent_id', $request->payment_intent_id)->get();
            foreach ($payments as $p) {
                $p->update(['status' => 'completed', 'paid_at' => now()]);
                $res = Reservation::find($p->reservation_id);
                if ($res) {
                    $newPaid = (float) $res->paid_amount + (float) $p->amount;
                    $newRemaining = (float) $res->remaining_amount - (float) $p->amount;
                    $res->update([
                        'paid_amount' => $newPaid,
                        'remaining_amount' => max(0, $newRemaining),
                        'status' => $newRemaining <= 0 ? 'confirmed' : $res->status,
                    ]);
                }
            }
            return response()->json(['ok' => true]);
        } catch (\Exception $e) {
            \Log::error('Confirm from-cart failed', ['error' => $e->getMessage()]);
            return response()->json(['message' => 'Onay işlenemedi', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Create Stripe checkout session
     */
    public function checkout(Request $request): JsonResponse
    {
        $validator = \Illuminate\Support\Facades\Validator::make($request->all(), [
            'reservation_id' => 'required|exists:reservations,id',
            'amount' => 'required|numeric|min:0.01',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $reservation = Reservation::findOrFail($request->reservation_id);

        if ($reservation->user_id !== $request->user()->id && !$request->user()->isAdmin()) {
            return response()->json([
                'message' => 'Unauthorized'
            ], 403);
        }

        try {
            $session = $this->getStripe()->checkout->sessions->create([
                'payment_method_types' => ['card'],
                'line_items' => [[
                    'price_data' => [
                        'currency' => 'eur',
                        'product_data' => [
                            'name' => "Rezervasyon: {$reservation->reservation_code}",
                        ],
                        'unit_amount' => $request->amount * 100, // Convert to cents
                    ],
                    'quantity' => 1,
                ]],
                'mode' => 'payment',
                'success_url' => config('app.frontend_url') . '/odeme-basarili?session_id={CHECKOUT_SESSION_ID}',
                'cancel_url' => config('app.frontend_url') . '/odeme-basarisiz',
                'metadata' => [
                    'reservation_id' => $reservation->id,
                    'user_id' => $request->user()->id,
                ],
            ]);

            // Create payment record
            $payment = Payment::create([
                'reservation_id' => $reservation->id,
                'user_id' => $request->user()->id,
                'payment_method' => 'stripe',
                'amount' => $request->amount,
                'status' => 'pending',
                'stripe_payment_intent_id' => $session->payment_intent,
            ]);

            return response()->json([
                'checkout_url' => $session->url,
                'session_id' => $session->id,
                'payment_id' => $payment->id,
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Payment creation failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get payment status
     */
    public function status(Request $request, int $id): JsonResponse
    {
        $payment = Payment::findOrFail($id);

        if ($payment->user_id !== $request->user()->id && !$request->user()->isAdmin()) {
            return response()->json([
                'message' => 'Unauthorized'
            ], 403);
        }

        // If Stripe payment, check status
        if ($payment->payment_method === 'stripe' && $payment->stripe_payment_intent_id) {
            try {
                $intent = $this->getStripe()->paymentIntents->retrieve($payment->stripe_payment_intent_id);
                $payment->update(['status' => $intent->status === 'succeeded' ? 'completed' : 'failed']);
            } catch (\Exception $e) {
                // Ignore errors
            }
        }

        return response()->json([
            'payment' => [
                'id' => $payment->id,
                'status' => $payment->status,
                'amount' => $payment->amount,
                'paid_at' => $payment->paid_at,
            ]
        ]);
    }

    /**
     * Retry payment
     */
    public function retry(Request $request, int $id): JsonResponse
    {
        $payment = Payment::findOrFail($id);

        if ($payment->user_id !== $request->user()->id && !$request->user()->isAdmin()) {
            return response()->json([
                'message' => 'Unauthorized'
            ], 403);
        }

        if ($payment->status !== 'failed') {
            return response()->json([
                'message' => 'Payment cannot be retried'
            ], 422);
        }

        // Create new checkout session
        return $this->checkout($request->merge(['reservation_id' => $payment->reservation_id, 'amount' => $payment->amount]));
    }

    /**
     * Get payment reports (Admin)
     */
    public function report(Request $request): JsonResponse
    {
        $query = Payment::with(['reservation.tour', 'user']);

        // Date range
        if ($request->has('date_from')) {
            $query->where('created_at', '>=', $request->date_from);
        }
        if ($request->has('date_to')) {
            $query->where('created_at', '<=', $request->date_to);
        }

        // Status filter
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // Payment method filter
        if ($request->has('payment_method')) {
            $query->where('payment_method', $request->payment_method);
        }

        $stats = [
            'total_payments' => (clone $query)->count(),
            'total_amount' => (clone $query)->where('status', 'completed')->sum('amount'),
            'by_status' => (clone $query)->selectRaw('status, count(*) as count, sum(amount) as total')
                ->groupBy('status')
                ->get(),
            'by_method' => (clone $query)->selectRaw('payment_method, count(*) as count, sum(amount) as total')
                ->where('status', 'completed')
                ->groupBy('payment_method')
                ->get(),
        ];

        return response()->json($stats);
    }

    /**
     * Process bank transfer payment (manual confirmation)
     */
    public function bankTransfer(Request $request): JsonResponse
    {
        $validator = \Illuminate\Support\Facades\Validator::make($request->all(), [
            'reservation_id' => 'required|exists:reservations,id',
            'easy_code' => 'required|string',
            'amount' => 'required|numeric|min:0.01',
            'reference' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $reservation = Reservation::where('easy_code', $request->easy_code)
            ->where('id', $request->reservation_id)
            ->firstOrFail();

        // Create pending payment
        $payment = Payment::create([
            'reservation_id' => $reservation->id,
            'user_id' => $reservation->user_id,
            'payment_method' => 'bank_transfer',
            'amount' => $request->amount,
            'status' => 'pending',
            'easy_code' => $request->easy_code,
            'bank_transfer_reference' => $request->reference,
        ]);

        return response()->json([
            'message' => 'Bank transfer payment recorded. Waiting for confirmation.',
            'payment' => [
                'id' => $payment->id,
                'easy_code' => $payment->easy_code,
                'status' => $payment->status,
            ]
        ]);
    }
}
