<?php

namespace App\Http\Controllers\Api\Flights;

use App\Http\Controllers\Controller;
use App\Models\FlightTicket;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class FlightTicketController extends Controller
{
    public function index(): JsonResponse
    {
        $tickets = FlightTicket::with('airline')->latest()->get();
        return response()->json(['data' => $tickets]);
    }

    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'airline_id' => 'nullable|exists:airlines,id',
            'type' => 'required|string',
            'is_direct' => 'required|boolean',
            'price' => 'required|numeric',
            'currency' => 'required|string',
            'slogan' => 'nullable|string',
            'outbound_date' => 'required|date',
            'outbound_time' => 'nullable|string',
            'outbound_departure_airport' => 'required|string',
            'outbound_arrival_airport' => 'required|string',
            'inbound_date' => 'nullable|date',
            'inbound_time' => 'nullable|string',
            'inbound_departure_airport' => 'nullable|string',
            'inbound_arrival_airport' => 'nullable|string',
            'is_active' => 'required|boolean',
            'brand_ids' => 'nullable|array',
            'agent_group_ids' => 'nullable|array',
        ]);

        $ticket = FlightTicket::create($validated);
        return response()->json(['data' => $ticket], 201);
    }

    public function show($id): JsonResponse
    {
        $ticket = FlightTicket::with('airline')->findOrFail($id);
        return response()->json(['data' => $ticket]);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $ticket = FlightTicket::findOrFail($id);
        
        $validated = $request->validate([
            'airline_id' => 'nullable|exists:airlines,id',
            'type' => 'string',
            'is_direct' => 'boolean',
            'price' => 'numeric',
            'currency' => 'string',
            'slogan' => 'nullable|string',
            'outbound_date' => 'date',
            'outbound_time' => 'nullable|string',
            'outbound_departure_airport' => 'string',
            'outbound_arrival_airport' => 'string',
            'inbound_date' => 'nullable|date',
            'inbound_time' => 'nullable|string',
            'inbound_departure_airport' => 'nullable|string',
            'inbound_arrival_airport' => 'nullable|string',
            'is_active' => 'boolean',
            'brand_ids' => 'nullable|array',
            'agent_group_ids' => 'nullable|array',
        ]);

        $ticket->update($validated);
        return response()->json(['data' => $ticket]);
    }

    public function destroy($id): JsonResponse
    {
        $ticket = FlightTicket::findOrFail($id);
        $ticket->delete();
        return response()->json(null, 24);
    }
}
