<?php

namespace App\Http\Controllers\API\Customers;

use App\Http\Controllers\Controller;
use App\Models\Reservation;
use App\Models\Payment;
use App\Models\Message;
use App\Models\Guest;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class CustomerPanelController extends Controller
{
    /**
     * Get dashboard data
     */
    public function dashboard(Request $request): JsonResponse
    {
        $user = $request->user();

        $stats = [
            'total_reservations' => Reservation::where('user_id', $user->id)->count(),
            'upcoming_reservations' => Reservation::where('user_id', $user->id)
                ->whereIn('status', ['approved', 'payment_pending', 'payment_completed', 'confirmed'])
                ->whereHas('tourDate', function ($q) {
                    $q->where('start_date', '>=', now());
                })
                ->count(),
            'total_spent' => Payment::where('user_id', $user->id)
                ->where('status', 'completed')
                ->sum('amount'),
            'loyalty_points' => $user->loyalty_points,
        ];

        $recentReservations = Reservation::with(['tour', 'tourDate'])
            ->where('user_id', $user->id)
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();

        return response()->json([
            'stats' => $stats,
            'recent_reservations' => $recentReservations,
        ]);
    }

    /**
     * Get user reservations
     */
    public function reservations(Request $request): JsonResponse
    {
        $reservations = Reservation::with(['tour', 'tourDate', 'payments'])
            ->where('user_id', $request->user()->id)
            ->orderBy('created_at', 'desc')
            ->paginate($request->get('per_page', 15));

        return response()->json($reservations);
    }

    /**
     * Get single reservation (for customer panel – meal options, guests, etc.)
     */
    public function showReservation(Request $request, int $id): JsonResponse
    {
        $reservation = Reservation::with(['tour', 'tourDate', 'guests', 'payments'])
            ->where('user_id', $request->user()->id)
            ->findOrFail($id);

        $tourDate = $reservation->tourDate;
        $mealOptions = $tourDate->meal_options ?? null;
        if (empty($mealOptions) && !empty($tourDate->meal_list)) {
            $mealOptions = ['categories' => [['key' => 'main', 'label' => 'Ana Menü', 'options' => $tourDate->meal_list]]];
        }

        return response()->json([
            'reservation' => $reservation,
            'meal_options' => $mealOptions,
        ]);
    }

    /**
     * Update a guest's meal selections (categoryKey => selectedOption)
     */
    public function updateGuestMealSelections(Request $request, int $id, int $guestId): JsonResponse
    {
        $reservation = Reservation::where('user_id', $request->user()->id)->findOrFail($id);
        $guest = $reservation->guests()->findOrFail($guestId);

        $validated = $request->validate([
            'meal_selections' => 'required|array',
            'meal_selections.*' => 'string',
        ]);

        $guest->meal_selections = $validated['meal_selections'];
        $guest->save();

        return response()->json([
            'message' => 'Yemek seçimleri güncellendi',
            'guest' => $guest->fresh(),
        ]);
    }

    /**
     * Get payment history
     */
    public function payments(Request $request): JsonResponse
    {
        $payments = Payment::with(['reservation.tour'])
            ->where('user_id', $request->user()->id)
            ->orderBy('created_at', 'desc')
            ->paginate($request->get('per_page', 15));

        return response()->json($payments);
    }

    /**
     * Get messages
     */
    public function messages(Request $request): JsonResponse
    {
        $messages = Message::with(['reservation.tour'])
            ->where('user_id', $request->user()->id)
            ->orderBy('created_at', 'desc')
            ->paginate($request->get('per_page', 15));

        return response()->json($messages);
    }

    /**
     * Send message
     */
    public function sendMessage(Request $request): JsonResponse
    {
        $validator = \Illuminate\Support\Facades\Validator::make($request->all(), [
            'subject' => 'required|string|max:255',
            'message' => 'required|string',
            'reservation_id' => 'nullable|exists:reservations,id',
            'type' => 'nullable|in:support,reservation,payment,general',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $message = Message::create([
            'user_id' => $request->user()->id,
            'reservation_id' => $request->reservation_id,
            'subject' => $request->subject,
            'message' => $request->message,
            'type' => $request->type ?? 'support',
            'status' => 'open',
        ]);

        return response()->json([
            'message' => 'Message sent successfully',
            'message_id' => $message->id,
        ], 201);
    }
}
