<?php

namespace App\Http\Controllers\Api\Activities;

use App\Http\Controllers\Controller;
use App\Models\Activity;
use App\Models\ActivityDate;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class ActivityController extends Controller
{
    /**
     * Get all activities
     */
    public function index(Request $request): JsonResponse
    {
        $activities = Activity::where('is_active', true)
            ->orderBy('order')
            ->get();

        return response()->json([
            'data' => $activities
        ]);
    }

    /**
     * Get activity by slug or ID
     */
    public function show($id): JsonResponse
    {
        $query = Activity::with(['pricingTiers', 'dates' => function($q) {
            $q->where('date', '>=', now())
              ->where('is_active', true)
              ->orderBy('date')
              ->orderBy('start_time');
        }]);

        if (is_numeric($id)) {
            $activity = $query->find($id);
        } else {
            $activity = $query->where('slug', $id)->first();
        }

        if (!$activity) {
            return response()->json(['message' => 'Activity not found'], 404);
        }

        return response()->json([
            'data' => $activity
        ]);
    }

    /**
     * Get available dates for an activity
     */
    public function dates(Request $request, $id): JsonResponse
    {
        $activity = Activity::findOrFail($id);
        
        // 1. Get explicit dates from DB
        $explicitDates = ActivityDate::where('activity_id', $id)
            ->where('date', '>=', now())
            ->orderBy('date')
            ->orderBy('start_time')
            ->get()
            ->get()
            ->keyBy('date'); // Key by date for easy lookup

        // Get pricing tiers
        $tiers = $activity->pricingTiers;


        // 2. Generate implicit dates if default_start_time is set
        $allDates = collect();
        
        if ($activity->default_start_time) {
            $startDate = now();
            // Generate dates for next 6 months
            $endDate = now()->addMonths(6);
            
            for ($date = $startDate; $date->lte($endDate); $date->addDay()) {
                $dateStr = $date->format('Y-m-d');
                
                // Check if explicit date exists for this day
                if ($explicitDates->has($dateStr)) {
                    $explicitDate = $explicitDates->get($dateStr);
                    // Only add if active (not closed)
                    if ($explicitDate->is_active) {
                        $allDates->push($explicitDate);
                    }
                } else {
                    // No explicit date, create implicit date
                    
                    // Create implicit pricing data from tiers
                    $pricingData = [];
                    foreach($tiers as $tier) {
                        $pricingData[$tier->id] = $tier->price;
                    }

                    $allDates->push([
                        'id' => 'implicit_' . $dateStr, // Virtual ID
                        'activity_id' => $activity->id,
                        'date' => $dateStr,
                        'start_time' => $activity->default_start_time,
                        'capacity' => 20, // Default capacity
                        'booked' => 0,
                        'is_active' => true,
                        'is_implicit' => true,
                        'pricing_data' => $pricingData,
                        'pricing_tiers' => $tiers // Include tiers definition for frontend convenience
                    ]);
                }
            }
        } else {
            // Fallback to only explicit active dates
             $allDates = $explicitDates->where('is_active', true)->values();
        }

        return response()->json([
            'data' => $allDates->values()
        ]);
    }

    /**
     * Calculate pricing
     */
    public function pricing(Request $request, $id): JsonResponse
    {
        $request->validate([
            'date_id' => 'required',
            'tiers' => 'required|array', // e.g. [{id: 1, quantity: 2}]
            'tiers.*.id' => 'required|integer',
            'tiers.*.quantity' => 'required|integer|min:0',
        ]);

        $activity = Activity::findOrFail($id);
        $totalPrice = 0;
        
        // Get Date Pricing
        $pricingData = [];
        
        if (str_starts_with($request->date_id, 'implicit_')) {
            // Implicit: Use default tier prices
            foreach($activity->pricingTiers as $tier) {
                $pricingData[$tier->id] = $tier->price;
            }
        } else {
            // Explicit: Use date specific prices (or fallback)
            $date = ActivityDate::findOrFail($request->date_id);
            if ($date->activity_id != $id) {
                 return response()->json(['message' => 'Invalid date'], 422);
            }
            $pricingData = $date->pricing_data ?? [];
            
            // Fallback for missing tier prices in explicit date to default tier prices
            foreach($activity->pricingTiers as $tier) {
                if (!isset($pricingData[$tier->id])) {
                    $pricingData[$tier->id] = $tier->price;
                }
            }
        }

        // Calculate Total
        foreach ($request->tiers as $item) {
            $tierId = $item['id'];
            $qty = $item['quantity'];
            
            if (isset($pricingData[$tierId])) {
                $totalPrice += floatval($pricingData[$tierId]) * $qty;
            }
        }

        return response()->json([
            'total_price' => $totalPrice,
            'currency' => 'EUR',
            'breakdown' => $request->tiers // meaningful breakdown could be added here
        ]);
    }
            'total' => $total,
            'breakdown' => [
                'adults' => [
                    'count' => $request->adults,
                    'unit_price' => str_starts_with($request->date_id, 'implicit_') ? $activity->base_price : $date->adult_price,
                    'total' => $adultPrice
                ],
                'children' => [
                    'count' => $request->input('children', 0),
                    'unit_price' => str_starts_with($request->date_id, 'implicit_') ? $activity->base_price * 0.5 : $date->child_price,
                    'total' => $childPrice
                ],
                'infants' => [
                    'count' => $request->input('infants', 0),
                    'unit_price' => str_starts_with($request->date_id, 'implicit_') ? 0 : $date->infant_price,
                    'total' => $infantPrice
                ]
            ]
        ]);
    }
}
