<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Slider;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class SliderController extends Controller
{
    /**
     * Display a listing of the sliders.
     */
    public function index(Request $request): JsonResponse
    {
        $query = Slider::with('brand');

        if ($request->has('brand_id')) {
            $query->where('brand_id', $request->brand_id);
        }

        $sliders = $query->orderBy('order')->orderBy('created_at', 'desc')->get();

        return response()->json([
            'data' => $sliders
        ]);
    }

    /**
     * Store a newly created slider in storage.
     */
    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'image' => 'required|string',
            'link' => 'nullable|string|max:500',
            'order' => 'integer',
            'is_active' => 'boolean',
            'brand_id' => 'required|exists:brands,id',
        ]);

        $slider = Slider::create($validated);

        return response()->json([
            'message' => 'Slider başarıyla oluşturuldu.',
            'data' => $slider->load('brand')
        ], 201);
    }

    /**
     * Display the specified slider.
     */
    public function show(Slider $slider): JsonResponse
    {
        return response()->json([
            'data' => $slider->load('brand')
        ]);
    }

    /**
     * Update the specified slider in storage.
     */
    public function update(Request $request, Slider $slider): JsonResponse
    {
        $validated = $request->validate([
            'name' => 'string|max:255',
            'image' => 'string',
            'link' => 'nullable|string|max:500',
            'order' => 'integer',
            'is_active' => 'boolean',
            'brand_id' => 'exists:brands,id',
        ]);

        $slider->update($validated);

        return response()->json([
            'message' => 'Slider başarıyla güncellendi.',
            'data' => $slider->load('brand')
        ]);
    }

    /**
     * Remove the specified slider from storage.
     */
    public function destroy(Slider $slider): JsonResponse
    {
        $slider->delete();

        return response()->json([
            'message' => 'Slider başarıyla silindi.'
        ]);
    }
}
