<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\File;

class SettingsController extends Controller
{
    /**
     * Get all settings or by group
     */
    public function index(Request $request): JsonResponse
    {
        $group = $request->get('group');
        
        if ($group) {
            $settings = Setting::getByGroup($group);
            // Eğer group için tek bir key varsa (örn: pdf_design), direkt value'yu döndür
            if (count($settings) === 1 && isset($settings[$group])) {
                return response()->json([
                    'settings' => $settings[$group],
                ]);
            }
            // Eğer birden fazla key varsa, hepsini döndür
            return response()->json([
                'settings' => $settings,
            ]);
        } else {
            $settings = Setting::all()->mapWithKeys(function ($setting) {
                return [$setting->key => Setting::castValue($setting->value, $setting->type)];
            })->toArray();
        }

        return response()->json([
            'settings' => $settings,
        ]);
    }

    /**
     * Get setting by key
     */
    public function show(string $key): JsonResponse
    {
        $setting = Setting::where('key', $key)->first();

        if (!$setting) {
            return response()->json([
                'message' => 'Setting not found'
            ], 404);
        }

        return response()->json([
            'setting' => [
                'key' => $setting->key,
                'value' => Setting::castValue($setting->value, $setting->type),
                'type' => $setting->type,
                'group' => $setting->group,
                'description' => $setting->description,
            ]
        ]);
    }

    /**
     * Update settings
     */
    public function update(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'settings' => 'required|array',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $updated = [];
        foreach ($request->settings as $key => $value) {
            $setting = Setting::where('key', $key)->first();
            
            // Determine type and group based on key
            $type = 'string';
            $group = 'general';
            
            if ($key === 'pdf_design' || $key === 'pdf_template') {
                $type = 'json';
                $group = $key;
            } elseif (in_array($key, ['stripe_key', 'stripe_secret', 'stripe_webhook_secret'])) {
                $type = 'string';
                $group = 'payment';
            } elseif (in_array($key, ['stripe_test_mode', 'stripe_3d_secure'])) {
                $type = 'boolean';
                $group = 'payment';
            }
            
            if ($setting) {
                // Update existing setting
                $valueToStore = $setting->type === 'json' 
                    ? json_encode($value) 
                    : ($setting->type === 'boolean' ? ($value ? 'true' : 'false') : (string)$value);
                
                $setting->update([
                    'value' => $valueToStore,
                    'type' => $type, // Update type if changed
                    'group' => $group, // Update group if changed
                ]);
                $updated[] = $key;
            } else {
                // Create new setting if doesn't exist
                $valueToStore = $type === 'json' 
                    ? json_encode($value) 
                    : ($type === 'boolean' ? ($value ? 'true' : 'false') : (string)$value);
                
                Setting::create([
                    'key' => $key,
                    'value' => $valueToStore,
                    'type' => $type,
                    'group' => $group,
                ]);
                $updated[] = $key;
            }
        }

        // If Stripe settings updated, update .env file (optional)
        if (isset($request->settings['stripe_key']) || 
            isset($request->settings['stripe_secret']) || 
            isset($request->settings['stripe_webhook_secret'])) {
            $this->updateEnvFile($request->settings);
        }

        return response()->json([
            'message' => 'Settings updated successfully',
            'updated' => $updated,
        ]);
    }

    /**
     * Update .env file with Stripe settings
     */
    private function updateEnvFile(array $settings): void
    {
        $envPath = base_path('.env');
        
        if (!File::exists($envPath)) {
            return;
        }

        $envContent = File::get($envPath);

        // Update Stripe keys
        if (isset($settings['stripe_key'])) {
            $envContent = preg_replace(
                '/^STRIPE_KEY=.*/m',
                'STRIPE_KEY=' . $settings['stripe_key'],
                $envContent
            );
        }

        if (isset($settings['stripe_secret'])) {
            $envContent = preg_replace(
                '/^STRIPE_SECRET=.*/m',
                'STRIPE_SECRET=' . $settings['stripe_secret'],
                $envContent
            );
        }

        if (isset($settings['stripe_webhook_secret'])) {
            $envContent = preg_replace(
                '/^STRIPE_WEBHOOK_SECRET=.*/m',
                'STRIPE_WEBHOOK_SECRET=' . $settings['stripe_webhook_secret'],
                $envContent
            );
        }

        File::put($envPath, $envContent);
    }

    /**
     * Test Stripe connection
     */
    public function testStripe(Request $request): JsonResponse
    {
        try {
            $stripeKey = $request->get('stripe_key') ?? Setting::get('stripe_key');
            $stripeSecret = $request->get('stripe_secret') ?? Setting::get('stripe_secret');

            if (!$stripeKey || !$stripeSecret) {
                return response()->json([
                    'success' => false,
                    'message' => 'Stripe keys not configured'
                ], 400);
            }

            $stripe = new \Stripe\StripeClient($stripeSecret);
            $account = $stripe->accounts->retrieve();

            return response()->json([
                'success' => true,
                'message' => 'Stripe connection successful',
                'account' => [
                    'id' => $account->id,
                    'email' => $account->email ?? null,
                    'country' => $account->country ?? null,
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Stripe connection failed: ' . $e->getMessage()
            ], 400);
        }
    }
}
