<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;

class ImageController extends Controller
{
    /**
     * Upload image file
     */
    public function upload(Request $request): JsonResponse
    {
        try {
            $request->validate([
                'image' => 'required|image|mimes:jpeg,jpg,png,webp|max:10240', // 10MB max
            ]);

            $file = $request->file('image');
            $originalName = $file->getClientOriginalName();
            $extension = $file->getClientOriginalExtension();
            
            // Unique filename
            $filename = time() . '_' . uniqid() . '.' . $extension;
            
            // Save directly to public/images (no storage link needed)
            $publicPath = public_path('images');
            
            // Ensure directory exists
            if (!is_dir($publicPath)) {
                mkdir($publicPath, 0755, true);
            }
            
            // Move file to public/images
            $file->move($publicPath, $filename);
            
            // Generate public URL
            // Direct access: http://localhost:8000/images/filename.jpg
            $url = '/images/' . $filename;
            
            // Full URL with domain
            $fullUrl = request()->getSchemeAndHttpHost() . $url;
            
            Log::info('Image uploaded successfully', [
                'filename' => $filename,
                'original_name' => $originalName,
                'public_path' => $publicPath . '/' . $filename,
                'relative_url' => $url,
                'full_url' => $fullUrl
            ]);
            
            return response()->json([
                'success' => true,
                'url' => $fullUrl,
                'filename' => $filename,
                'path' => 'images/' . $filename
            ]);
            
        } catch (\Exception $e) {
            Log::error('Image upload error: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            return response()->json([
                'success' => false,
                'error' => 'Görsel yüklenirken hata oluştu: ' . $e->getMessage()
            ], 500);
        }
    }
}
