<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Brand;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;

class BrandController extends Controller
{
    /**
     * Get all brands
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $query = Brand::query();

            if ($request->has('is_active')) {
                $query->where('is_active', $request->is_active === 'true');
            }

            $brands = $query->orderBy('name')->get();

            return response()->json([
                'data' => $brands
            ]);
        } catch (\Exception $e) {
            Log::error('Brands index error: ' . $e->getMessage());
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Get brand by ID
     */
    public function show(int $id): JsonResponse
    {
        try {
            $brand = Brand::findOrFail($id);
            return response()->json(['data' => $brand]);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Get brand by API key (for external systems)
     */
    public function getByApiKey(Request $request): JsonResponse
    {
        try {
            $apiKey = $request->header('X-API-Key') ?: $request->get('api_key');
            
            if (!$apiKey) {
                return response()->json(['error' => 'API key is required'], 401);
            }

            $brand = Brand::where('api_key', $apiKey)
                ->where('is_active', true)
                ->first();

            if (!$brand) {
                return response()->json(['error' => 'Invalid API key'], 401);
            }

            return response()->json([
                'data' => [
                    'id' => $brand->id,
                    'name' => $brand->name,
                    'domain' => $brand->domain,
                    'slug' => $brand->slug,
                    'logo' => $brand->logo,
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Brand API key verification error: ' . $e->getMessage());
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Verify API key and return brand info
     */
    public function verifyApiKey(Request $request): JsonResponse
    {
        try {
            $apiKey = $request->header('X-API-Key') ?: $request->get('api_key');
            $domain = $request->get('domain');
            
            if (!$apiKey) {
                return response()->json(['error' => 'API key is required'], 401);
            }

            $brand = Brand::where('api_key', $apiKey)
                ->where('is_active', true)
                ->first();

            if (!$brand) {
                return response()->json([
                    'valid' => false,
                    'error' => 'Invalid API key'
                ], 401);
            }

            // Domain eşleşmesi kontrolü (opsiyonel ama önerilir)
            if ($domain && $brand->domain !== $domain) {
                return response()->json([
                    'valid' => false,
                    'error' => 'Domain mismatch'
                ], 403);
            }

            return response()->json([
                'valid' => true,
                'data' => [
                    'id' => $brand->id,
                    'name' => $brand->name,
                    'domain' => $brand->domain,
                    'slug' => $brand->slug,
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('API key verification error: ' . $e->getMessage());
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }
}
