# 🚀 Sonraki Adımlar

## ✅ Tamamlananlar

1. ✅ Laravel proje yapısı hazırlandı
2. ✅ API route yapısı oluşturuldu
3. ✅ CORS yapılandırması (React frontend için)
4. ✅ Sanctum yapılandırması
5. ✅ API endpoint dokümantasyonu
6. ✅ Composer.json (gerekli paketler listelendi)

## 📋 Yapılacaklar

### 1. PHP ve Composer Kurulumu

```bash
# macOS için
brew install php
curl -sS https://getcomposer.org/installer | php
sudo mv composer.phar /usr/local/bin/composer
```

Detaylı kurulum için: `INSTALLATION.md` dosyasına bakın.

### 2. Laravel Projesi Oluşturma

```bash
cd backend
composer create-project laravel/laravel . --prefer-dist
```

Veya mevcut dosyaları Laravel projesine kopyalayın.

### 3. Paketlerin Kurulumu

```bash
composer require laravel/sanctum
composer require laravel/cashier
composer require spatie/laravel-permission
composer require spatie/laravel-multitenancy
composer require spatie/laravel-media-library
composer require barryvdh/laravel-dompdf
composer require simplesoftwareio/simple-qrcode
composer require stripe/stripe-php
```

### 4. Veritabanı Migration'ları

Veritabanı şeması için migration dosyaları oluşturulacak:
- Brands (Markalar)
- Tours (Turlar)
- TourDates (Tur Tarihleri)
- Reservations (Rezervasyonlar)
- Payments (Ödemeler)
- Guests (Misafirler)
- Rooms (Odalar)
- vb.

### 5. Model ve Controller'lar

- Models: User, Tour, Reservation, Payment, vb.
- Controllers: AuthController, TourController, ReservationController, vb.
- Services: PaymentService, ReservationService, vb.

### 6. Frontend Entegrasyonu

React frontend'i API'ye bağlamak için:

1. API base URL'i ayarlayın:
```typescript
// src/config/api.ts
export const API_BASE_URL = 'http://localhost:8000/api';
```

2. API service oluşturun:
```typescript
// src/services/api.ts
import axios from 'axios';

const api = axios.create({
  baseURL: API_BASE_URL,
  headers: {
    'Content-Type': 'application/json',
  },
});

// Token interceptor
api.interceptors.request.use((config) => {
  const token = localStorage.getItem('auth_token');
  if (token) {
    config.headers.Authorization = `Bearer ${token}`;
  }
  return config;
});
```

3. Mevcut `data.ts` yerine API çağrıları yapın:
```typescript
// Önceki: import { TOURS } from './data';
// Yeni: 
const { data } = await api.get('/tours');
```

## 📁 Dosya Yapısı

```
backend/
├── README.md                    ✅ Oluşturuldu
├── INSTALLATION.md              ✅ Oluşturuldu
├── PROJECT_STRUCTURE.md        ✅ Oluşturuldu
├── API_ENDPOINTS.md            ✅ Oluşturuldu
├── composer.json                ✅ Oluşturuldu
├── routes/
│   └── api.php                 ✅ Oluşturuldu
└── config/
    ├── cors.php                ✅ Oluşturuldu
    └── sanctum.php             ✅ Oluşturuldu
```

## 🔗 Frontend Entegrasyonu

Mevcut React frontend'iniz (`avusturya gezi/`) şu an statik veri kullanıyor (`data.ts`). 
API'ye geçiş için:

1. **API Service Layer** oluşturun
2. **State Management** (Context API veya Zustand)
3. **API çağrılarını** mevcut component'lere entegre edin

## 🎯 Öncelik Sırası

1. **PHP/Composer kurulumu** (Gerekli)
2. **Laravel projesi oluşturma**
3. **Veritabanı migration'ları** (Kritik)
4. **Authentication API** (İlk çalışan endpoint)
5. **Tours API** (Frontend'in ihtiyacı)
6. **Reservations API** (Ana işlevsellik)
7. **Payments API** (Stripe entegrasyonu)

## 📞 Yardım

Herhangi bir adımda takılırsanız:
1. `INSTALLATION.md` dosyasına bakın
2. Laravel dokümantasyonuna bakın: https://laravel.com/docs
3. API endpoint'ler için `API_ENDPOINTS.md` dosyasına bakın

## 🚀 Hızlı Başlangıç

```bash
# 1. PHP ve Composer kur
brew install php
curl -sS https://getcomposer.org/installer | php

# 2. Laravel projesi oluştur
cd backend
composer create-project laravel/laravel . --prefer-dist

# 3. Paketleri kur
composer require laravel/sanctum laravel/cashier spatie/laravel-permission

# 4. Environment ayarla
cp .env.example .env
php artisan key:generate

# 5. Veritabanı oluştur ve migrate et
php artisan migrate

# 6. Server'ı başlat
php artisan serve
```

API artık `http://localhost:8000/api` adresinde çalışıyor! 🎉
